/*************************************************************************
*    CompuCell - A software framework for multimodel simulations of     *
* biocomplexity problems Copyright (C) 2003 University of Notre Dame,   *
*                             Indiana                                   *
*                                                                       *
* This program is free software; IF YOU AGREE TO CITE USE OF CompuCell  *
*  IN ALL RELATED RESEARCH PUBLICATIONS according to the terms of the   *
*  CompuCell GNU General Public License RIDER you can redistribute it   *
* and/or modify it under the terms of the GNU General Public License as *
*  published by the Free Software Foundation; either version 2 of the   *
*         License, or (at your option) any later version.               *
*                                                                       *
* This program is distributed in the hope that it will be useful, but   *
*      WITHOUT ANY WARRANTY; without even the implied warranty of       *
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    *
*             General Public License for more details.                  *
*                                                                       *
*  You should have received a copy of the GNU General Public License    *
*     along with this program; if not, write to the Free Software       *
*      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
*************************************************************************/



#include <CompuCell3D/Potts3D/CellInventory.h>
#include <CompuCell3D/Automaton/Automaton.h>

#include <CompuCell3D/Simulator.h>
#include <CompuCell3D/Potts3D/Cell.h>
#include <CompuCell3D/Potts3D/Potts3D.h>
#include <CompuCell3D/plugins/NeighborTracker/NeighborTrackerPlugin.h>

using namespace CompuCell3D;
#include <BasicUtils/BasicString.h>
#include <BasicUtils/BasicException.h>
#include <BasicUtils/BasicClassAccessor.h>
#include <string>
#include <fstream>
#include <sstream>

using namespace std;


#define EXP_STL

#include "TCS.h"

TCS::TCS(){};

TCS::~TCS(){}


void TCS::init(Simulator *_simulator, CC3DXMLElement *_xmlData){
	potts = _simulator->getPotts();
	cellInventoryPtr = & potts->getCellInventory();
	setParameters(_simulator,_xmlData);
}

void TCS::extraInit(Simulator *simulator){
	// Make sure all necessary plugins are loaded
	bool pluginAlreadyRegisteredFlag;
	// Load neighbortracker and initiate when needed
	NeighborTrackerPlugin * neighborTrackerPluginPtr=(NeighborTrackerPlugin*)(Simulator::pluginManager.get("NeighborTracker",&pluginAlreadyRegisteredFlag));
	if (!pluginAlreadyRegisteredFlag){neighborTrackerPluginPtr->init(simulator);}
	ASSERT_OR_THROW("NeighborTracker plugin not initialized!", neighborTrackerPluginPtr);
	neighborTrackerAccessorPtr=neighborTrackerPluginPtr->getNeighborTrackerAccessorPtr();
	ASSERT_OR_THROW("neighborAccessorPtr  not initialized!", neighborTrackerAccessorPtr);
}

void TCS::start(){
	// Create output files if data will be saved during the simulation
	if (!save){ return;}
	CellInventory::cellInventoryIterator cInvItr;
	CellG * cell;
	ostringstream str;
	ofstream f;
	for(cInvItr=cellInventoryPtr->cellInventoryBegin() ; cInvItr !=cellInventoryPtr->cellInventoryEnd() ;++cInvItr ){
		cell=cellInventoryPtr->getCell(cInvItr);
		if ((((int)cell->type)==stalkID) || (((int)cell->type)==tipID)){
			str.str("");
			str<<fn<<"_NICD_"<<cell->id<<".data";				
			f.open(str.str().c_str());
			f << "MCS\tNICD\n";
			f.close();
		}
	}
	
}

void TCS::setNotch(){
	// Set Notch intracellular domain levels for each cell
	CellInventory::cellInventoryIterator cInvItr;
	CellG * cell;
	CellG * neighbor;
	NeighborTracker * neighborTracker;
	set<NeighborSurfaceData>::iterator neighborItr;
	double n;
	for(cInvItr=cellInventoryPtr->cellInventoryBegin() ; cInvItr !=cellInventoryPtr->cellInventoryEnd() ;++cInvItr ){
		cell=cellInventoryPtr->getCell(cInvItr);
		if (cell){
			n = 0.0;
			neighborTracker = neighborTrackerAccessorPtr->get(cell->extraAttribPtr);
			// calculate the level of Delta the cell senses
			for(neighborItr = neighborTracker->cellNeighbors.begin() ; neighborItr != neighborTracker->cellNeighbors.end() ; ++neighborItr){
				neighbor = neighborItr->neighborAddress;
				if(neighbor)
					n += decd[neighbor->type]*neighborItr->commonSurfaceArea;
			}
			// calculate the notch intracellular domain level
			notch[cell->id] = n*necd[cell->type]/cell->volume;
		}
	}
}

void TCS::updateTypes(){
	// update cell types (without saving any data)
	CellInventory::cellInventoryIterator cInvItr;
	CellG * cell;
	ostringstream str;
	ofstream f;
	for(cInvItr=cellInventoryPtr->cellInventoryBegin() ; cInvItr !=cellInventoryPtr->cellInventoryEnd() ;++cInvItr ){
		cell=cellInventoryPtr->getCell(cInvItr);
		// compare notch intracellular domain levels with the threshold for tip and stalk cells
		if ((((int)cell->type)==stalkID) || (((int)cell->type)==tipID)){
			if (notch[cell->id] > th)
				cell->type = stalkID;
			else
				cell->type = tipID;
		}
	}
}

void TCS::updateTypes(int currentStep){
	// update cell types and write notch intracellular domain levels to file
	CellInventory::cellInventoryIterator cInvItr;
	CellG * cell;
	ostringstream str;
	ofstream f;
	for(cInvItr=cellInventoryPtr->cellInventoryBegin() ; cInvItr !=cellInventoryPtr->cellInventoryEnd() ;++cInvItr ){
		cell=cellInventoryPtr->getCell(cInvItr);
		// compare notch intracellular domain levels with the threshold for tip and stalk cells		
		if ((((int)cell->type)==stalkID) || (((int)cell->type)==tipID)){
			if (notch[cell->id] > th)
				cell->type = stalkID;
			else
				cell->type = tipID;
			str.str("");
			str<<fn<<"_NICD_"<<cell->id<<".data";				
			f.open(str.str().c_str(),ios::out | ios::app);
			f << currentStep << "\t" << notch[cell->id] << "\n";
			f.close();
		}
	}
}

void TCS::step(const unsigned int currentStep) {
	setNotch();
	if ((save) && ((currentStep % savefreq) == 0)) {updateTypes(currentStep);}
	else {updateTypes();}
}


void TCS::finish(){}

//** Extra function to read xml **//
void TCS::setParameters(Simulator *_simulator, CC3DXMLElement *_xmlData){
	Automaton * automaton=potts->getAutomaton();
	CC3DXMLElement *dataXMLElement;
	int type;
	dataXMLElement=_xmlData->getFirstElement("tip");
	ASSERT_OR_THROW("You must provide tip cell properties!", dataXMLElement);
	tipID = (int)automaton->getTypeId(dataXMLElement->getAttribute("type"));
	cout << "tip cell has type " << tipID << endl;
	necd[tipID] = dataXMLElement->getAttributeAsDouble("notch");
	decd[tipID] = dataXMLElement->getAttributeAsDouble("dll");
	dataXMLElement=_xmlData->getFirstElement("stalk");
	ASSERT_OR_THROW("You must provide stalk cell properties!", dataXMLElement);
	stalkID = (int)automaton->getTypeId(dataXMLElement->getAttribute("type"));
	cout << "stalk cell has type " << stalkID << endl;
	necd[stalkID] = dataXMLElement->getAttributeAsDouble("notch");
	decd[stalkID] = dataXMLElement->getAttributeAsDouble("dll");
	dataXMLElement=_xmlData->getFirstElement("threshold");
	ASSERT_OR_THROW("You must provide a threshold!", dataXMLElement);
	th = dataXMLElement->getDouble();
	CC3DXMLElement *savedataXMLElement;
	savedataXMLElement=_xmlData->getFirstElement("save");
	if (savedataXMLElement) {
		cout << "wat?" << endl;
		fn = savedataXMLElement->getText();
		save = true;
		if (savedataXMLElement->findAttribute("freq")){savefreq=savedataXMLElement->getAttributeAsInt("freq");}
		else{savefreq=1;}		
	}
	else{save = false;}
}

std::string TCS::toString(){return "TCS";}





