#include <CompuCell3D/Field3D/Field3D.h>
#include <CompuCell3D/Field3D/WatchableField3D.h>
#include <CompuCell3D/Potts3D/Potts3D.h>
#include <CompuCell3D/Simulator.h>
#include <CompuCell3D/Automaton/Automaton.h>
using namespace CompuCell3D;

#include <BasicUtils/BasicString.h>
#include <BasicUtils/BasicException.h>
#include <set>
#include "ConnectivityFroNoPlugin.h"


ConnectivityFroNoPlugin::ConnectivityFroNoPlugin() : penalty(0.0) ,potts(0) ,numberOfNeighbors(8)
{
   n.assign(numberOfNeighbors,Point3D(0,0,0)); //allocates memory for vector of neighbors points
   offsetsIndex.assign(numberOfNeighbors,0);//allocates memory for vector of offsetsIndexes
}

ConnectivityFroNoPlugin::~ConnectivityFroNoPlugin() {
}

void ConnectivityFroNoPlugin::init(Simulator *simulator, CC3DXMLElement *_xmlData) {
  
  potts=simulator->getPotts();
  potts->registerEnergyFunction(this);
  simulator->registerSteerableObject(this);
  update(_xmlData,true);

     //here we initialize offsets for neighbors (up to second nearest) in 2D
   initializeNeighborsOffsets();
}

void ConnectivityFroNoPlugin::update(CC3DXMLElement *_xmlData, bool _fullInitFlag){

	if(potts->getDisplayUnitsFlag()){
		Unit energyUnit=potts->getEnergyUnit();




		CC3DXMLElement * unitsElem=_xmlData->getFirstElement("Units"); 
		if (!unitsElem){ //add Units element
			unitsElem=_xmlData->attachElement("Units");
		}

		if(unitsElem->getFirstElement("PenaltyUnit")){
			unitsElem->getFirstElement("PenaltyUnit")->updateElementValue(energyUnit.toString());
		}else{
			CC3DXMLElement * energyElem = unitsElem->attachElement("PenaltyUnit",energyUnit.toString());
		}
	}

	if(_xmlData){
		penalty=_xmlData->getFirstElement("Penalty")->getDouble();
	}
}

void ConnectivityFroNoPlugin::addUnique(CellG* cell,std::vector<CellG*> & _uniqueCells)
{

   // Medium does not count
   for (unsigned int i = 0; i < _uniqueCells.size(); i++)
   {
      if (_uniqueCells[i] == cell)
          return;
   }
   _uniqueCells.push_back(cell);
}

//modified version of connectivity algorithm - Tinri Aegerter
//added check for frozen pixels - Margriet Palm
double ConnectivityFroNoPlugin::changeEnergy(const Point3D &pt,
                                  const CellG *newCell,
                                   const CellG *oldCell) {

   if (!oldCell) return 0;
   std::vector<CellG*> uniqueCells;
   std::vector<Point3D> n(numberOfNeighbors,Point3D());
   // uniqueCells.clear();
   Neighbor neighbor;

   //rule 1: go over first neighrest neighbors and check if newCell has 1st neighrest neioghbotsbelongin to the same cell   
   bool firstFlag=false;
   unsigned int firstMaxIndex=boundaryStrategy->getMaxNeighborIndexFromNeighborOrder(1);
   for(int i=0 ; i <=firstMaxIndex ; ++i ){
      neighbor=boundaryStrategy->getNeighborDirect(const_cast<Point3D&>(pt),i);
      if(!neighbor.distance){
         //if distance is 0 then the neighbor returned is invalid
         continue;
      }
      if(potts->getCellFieldG()->get(neighbor.pt)==newCell){
         firstFlag=true;
         break;
      }
   }  

   if(!firstFlag){
      return penalty;
   }


   // Immediate neighbors

      for(int i=0 ; i < numberOfNeighbors ; ++i ){
         neighbor=boundaryStrategy->getNeighborDirect(const_cast<Point3D&>(pt),offsetsIndex[i]);
         if(!neighbor.distance){
         //if distance is 0 then the neighbor returned is invalid
         continue;
         }

         n[i]=neighbor.pt;
      }

   //rule 2: Count collisions between pixels belonging to different cells. If the collision count is greater than 2 reject the spin flip
   // mediumFlag = false;
      vector<Point3D> neighborsOld;
      vector<Point3D> neighborsNew;
      bool difFlag = false;

   int collisioncount = 0;
   for (unsigned int i = 0; i < numberOfNeighbors; i++)
   {
      if (
            ((potts->getCellFieldG()->get(n[i]) == oldCell) || (potts->getCellFieldG()->get(n[(i+1) % numberOfNeighbors]) == oldCell))
              &&
            (potts->getCellFieldG()->get(n[i]) !=potts->getCellFieldG()->get(n[(i+1) % numberOfNeighbors]))
         )
      {
        addUnique(potts->getCellFieldG()->get(n[i]),uniqueCells);
        addUnique(potts->getCellFieldG()->get(n[(i+1)%numberOfNeighbors]),uniqueCells);
	collisioncount++;
      }
      if (potts->getCellFieldG()->get(n[i]) == oldCell) neighborsOld.push_back(n[i]);
      else if (potts->getCellFieldG()->get(n[i]) == newCell) neighborsNew.push_back(n[i]);
      else difFlag = true;
   }
   bool frozenPixel = false;
   if (collisioncount == 2) return 0;  // Accept
   else if (!difFlag){
	   // At this moment the flip may be rejected because of connectivity
	   // Yet, if the target pixel is a frozen pixel we should be more
	   // careful about rejecting it. Thus, we check for a specific configuration:
	   // o n n		n = newCell
	   // n o n
	   // o n n		o = oldCell
	   // If we find this structure, check if the oldCell pixel in the middle
	   // is connected via the other side:
	   // - o n n
	   // o n o n
	   // - o n n
	   // If this is the case, we have a frozen pixel for which the flip
	   // shouldn't be rejected
	   // Check if the pixel could be a frozen pixel
	   if ((neighborsOld.size() == 2) && (neighborsNew.size() == 6) && (newCell)){
		   Point3D n1 = neighborsOld[0];
		   Point3D n2 = neighborsOld[1];
   		   // check if the old cell pixels are on the right spot
   		   if (!((n1.x == pt.x) || (n1.y == pt.y) || (n2.x == pt.x) || (n2.y == pt.y))
   				   &&
   			  ((n1.x == n2.x) || (n1.y == n2.y))){
   			   // this may be a Frozen Pixel, check if it is connected
   			   Point3D p;
   			   if (n1.x == n2.x) {
   				   p.x = n1.x+((n1.x > pt.x) ? 1 : -1);
   				   p.y = pt.y;
   			   }
   			   if (n1.y == n2.y){
   				   p.x = pt.x;
   				   p.y = n1.y+((n1.y > pt.y) ? 1 : -1);
   			   }
   			   if (potts->getCellFieldG()->get(p)){
   				   if (potts->getCellFieldG()->get(p) == oldCell) frozenPixel = true;
   			   }
   		   }
   	   }
      }
   	   if (frozenPixel) {
   		   return 0;
   	   }
   	   else {
   		   return penalty;
   	   }
}

void ConnectivityFroNoPlugin::orderNeighborsClockwise
(Point3D & _midPoint, const std::vector<Point3D> & _offsets, std::vector<int> & _offsetsIndex)
{
      //this function maps indexes of neighbors as given by boundary strategy to indexes of neighbors which are in clockwise order
      //that is when iterating over list of neighbors using boundaryStrategy API you can retrieve neighbors in clocwise order if you use
      //neighbor=boundaryStrategy->getNeighborDirect(const_cast<Point3D&>(pt),offsetsIndex[i]) call
      Neighbor neighbor;
      for(unsigned int nIdx=0 ; nIdx <= maxNeighborIndex ; ++nIdx ){
         neighbor=boundaryStrategy->getNeighborDirect(const_cast<Point3D&>(_midPoint),nIdx);
         if(!neighbor.distance){
         //if distance is 0 then the neighbor returned is invalid
         continue;
         }
         for(int i = 0 ; i < numberOfNeighbors ; ++i){
            if(_midPoint+_offsets[i] == neighbor.pt){
               _offsetsIndex[i]=nIdx;
            }
         }
      }
}


void ConnectivityFroNoPlugin::initializeNeighborsOffsets(){
   
   Dim3D fieldDim=potts->getCellFieldG()->getDim();
   vector<Point3D> offsets;

   offsets.assign(numberOfNeighbors,Point3D(0,0,0));//allocates memory for vector of offsets 

   boundaryStrategy=BoundaryStrategy::getInstance();
   maxNeighborIndex=0;

   maxNeighborIndex=boundaryStrategy->getMaxNeighborIndexFromDepth(1.45);//second nearest neighbor

   ASSERT_OR_THROW("This plugin will only work for 2D simulations i.e. one lattice dimansion must be equal to 1 Your simulations appears to be 3D", !(fieldDim.x>1 && fieldDim.y>1 && fieldDim.z>1 ) );

   //here we define neighbors  offsets in the "clockwise order"
   if(fieldDim.x==1){
      //clockwise offsets
      offsets[0]=Point3D(0,0,-1);
      offsets[1]=Point3D(0,-1,-1);
      offsets[2]=Point3D(0,-1,0);
      offsets[3]=Point3D(0,-1,1);
      offsets[4]=Point3D(0,0,1);
      offsets[5]=Point3D(0,1,1);
      offsets[6]=Point3D(0,1,0);
      offsets[7]=Point3D(0,1,-1);

      Point3D midPoint(0, fieldDim.y/2, fieldDim.z/2);
      orderNeighborsClockwise(midPoint , offsets , offsetsIndex);

   }

   if(fieldDim.y==1){

      offsets[0]=Point3D(0,0,-1);
      offsets[1]=Point3D(-1,0,-1);
      offsets[2]=Point3D(-1,0,0);
      offsets[3]=Point3D(-1,0,1);
      offsets[4]=Point3D(0,0,1);
      offsets[5]=Point3D(1,0,1);
      offsets[6]=Point3D(1,0,0);
      offsets[7]=Point3D(1,0,-1);

      Point3D midPoint(fieldDim.x/2, 0, fieldDim.z/2);
      orderNeighborsClockwise(midPoint , offsets , offsetsIndex);



   }
   
   if(fieldDim.z==1){
      offsets[0]=Point3D(0,-1,0);
      offsets[1]=Point3D(-1,-1,0);
      offsets[2]=Point3D(-1,0,0);
      offsets[3]=Point3D(-1,1,0);
      offsets[4]=Point3D(0,1,0);
      offsets[5]=Point3D(1,1,0);
      offsets[6]=Point3D(1,0,0);
      offsets[7]=Point3D(1,-1,0);
      
      Point3D midPoint(fieldDim.x/2, fieldDim.y/2, 0);
      orderNeighborsClockwise(midPoint , offsets , offsetsIndex);
   }

}

std::string ConnectivityFroNoPlugin::toString(){
	return "ConnectivityFroNo";
}
std::string ConnectivityFroNoPlugin::steerableName(){
   return toString();
}
